package com.castlabs.sdk.castdemo;

import android.app.Dialog;
import android.content.Context;
import android.graphics.Color;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.util.SparseArray;
import android.view.View;
import android.widget.ArrayAdapter;
import android.widget.SeekBar;
import android.widget.Spinner;
import android.widget.TextView;

import com.castlabs.subtitles.presentation.SubtitlesStyle;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Locale;

import static com.castlabs.subtitles.presentation.SubtitlesStyle.EDGE_TYPE_DEPRESSED;
import static com.castlabs.subtitles.presentation.SubtitlesStyle.EDGE_TYPE_DROP_SHADOW;
import static com.castlabs.subtitles.presentation.SubtitlesStyle.EDGE_TYPE_NONE;
import static com.castlabs.subtitles.presentation.SubtitlesStyle.EDGE_TYPE_OUTLINE;
import static com.castlabs.subtitles.presentation.SubtitlesStyle.EDGE_TYPE_RAISED;

public class SubtitleStyleDialog extends Dialog {

	public interface OnDialogSubmitListener {
		void onDialogSubmit(SubtitleStyleDialog dialog);
	}

	public final Spinner spinnerBackgroundColor;
	public final Spinner spinnerForegroundColor;
	public final Spinner spinnerEdgeColor;
	public final Spinner spinnerEdgeType;
	public final Spinner spinnerWindowColor;
	public final Spinner spinnerFontFamily;
	public final TextView textViewScale;
	public final SeekBar seekBarFontScale;
	public final ArrayAdapter<String> colorsAdapter;
	public final ArrayAdapter<String> edgeTypesAdapter;
	private final HashMap<String, Integer> colorNameToValueMap;
	private final SparseArray<String> colorValueToNameMap;
	private final HashMap<String, Integer> edgeNameToTypeMap;
	private final SparseArray<String> edgeTypeToNameMap;
	private final ArrayList<String> fontFamilyList;
	private final ArrayAdapter<String> fontFamilyAdapter;
	private OnDialogSubmitListener dialogSubmitListener;

	public SubtitleStyleDialog(@NonNull Context context, @Nullable SubtitlesStyle subtitlesStyle) {
		super(context);

		setContentView(R.layout.dialog_subtitles_style);
		spinnerBackgroundColor = (Spinner) findViewById(R.id.spinner_background_colors);
		spinnerForegroundColor = (Spinner) findViewById(R.id.spinner_foreground_colors);
		spinnerEdgeColor = (Spinner) findViewById(R.id.spinner_edge_colors);
		spinnerEdgeType = (Spinner) findViewById(R.id.spinner_edge_type);
		spinnerWindowColor = (Spinner) findViewById(R.id.spinner_window_colors);
		textViewScale = (TextView) findViewById(R.id.textview_font_scale);
		seekBarFontScale = (SeekBar) findViewById(R.id.seek_bar_font_scale);
		spinnerFontFamily = (Spinner) findViewById(R.id.spinner_font_family);

		colorNameToValueMap = new LinkedHashMap<>();
		colorValueToNameMap = new SparseArray<>();
		colorNameToValueMap.put("Cyan", Color.CYAN);
		colorNameToValueMap.put("Red", Color.RED);
		colorNameToValueMap.put("Black", Color.BLACK);
		colorNameToValueMap.put("Green", Color.GREEN);
		colorNameToValueMap.put("Blue", Color.BLUE);
		colorNameToValueMap.put("White", Color.WHITE);
		colorNameToValueMap.put("Magenta", Color.MAGENTA);
		colorNameToValueMap.put("Yellow", Color.YELLOW);
		colorNameToValueMap.put("Transparent", Color.TRANSPARENT);
		for (String color : colorNameToValueMap.keySet()){
			colorValueToNameMap.put(colorNameToValueMap.get(color), color);
		}

		colorsAdapter = new ArrayAdapter<>(context, android.R.layout.simple_spinner_dropdown_item);
		colorsAdapter.addAll(colorNameToValueMap.keySet());

		edgeNameToTypeMap = new HashMap<>();
		edgeTypeToNameMap = new SparseArray<>();
		edgeNameToTypeMap.put("None", EDGE_TYPE_NONE);
		edgeNameToTypeMap.put("Outline", EDGE_TYPE_OUTLINE);
		edgeNameToTypeMap.put("Drop Shadow", EDGE_TYPE_DROP_SHADOW);
		edgeNameToTypeMap.put("Raised", EDGE_TYPE_RAISED);
		edgeNameToTypeMap.put("Depressed", EDGE_TYPE_DEPRESSED);
		for (String type : edgeNameToTypeMap.keySet()){
			edgeTypeToNameMap.put(edgeNameToTypeMap.get(type), type);
		}

		edgeTypesAdapter = new ArrayAdapter<>(context, android.R.layout.simple_spinner_dropdown_item);
		edgeTypesAdapter.addAll(edgeNameToTypeMap.keySet());

		fontFamilyList = new ArrayList<>();
		fontFamilyList.add("sans-serif");
		fontFamilyList.add("serif");
		fontFamilyList.add("monospace");

		fontFamilyAdapter = new ArrayAdapter<>(context, android.R.layout.simple_spinner_dropdown_item, fontFamilyList);

		seekBarFontScale.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
			@Override
			public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
				double scale = progress / 25.0;
				textViewScale.setText(String.format(Locale.US, "%.2f", scale));
			}

			@Override
			public void onStartTrackingTouch(SeekBar seekBar) {
			}

			@Override
			public void onStopTrackingTouch(SeekBar seekBar) {
			}
		});

		spinnerBackgroundColor.setAdapter(colorsAdapter);
		spinnerForegroundColor.setAdapter(colorsAdapter);
		spinnerEdgeColor.setAdapter(colorsAdapter);
		spinnerWindowColor.setAdapter(colorsAdapter);
		spinnerEdgeType.setAdapter(edgeTypesAdapter);
		spinnerFontFamily.setAdapter(fontFamilyAdapter);

		// Set initial values
		if (subtitlesStyle == null) {
			spinnerBackgroundColor.setSelection(0);
			spinnerForegroundColor.setSelection(1);
			spinnerEdgeColor.setSelection(2);
			spinnerWindowColor.setSelection(3);
			seekBarFontScale.setProgress(25);
		} else {
			spinnerBackgroundColor.setSelection(colorsAdapter.getPosition(colorValueToNameMap.get(subtitlesStyle.backgroundColor)));
			spinnerForegroundColor.setSelection(colorsAdapter.getPosition(colorValueToNameMap.get(subtitlesStyle.foregroundColor)));
			spinnerEdgeColor.setSelection(colorsAdapter.getPosition(colorValueToNameMap.get(subtitlesStyle.edgeColor)));
			spinnerWindowColor.setSelection(colorsAdapter.getPosition(colorValueToNameMap.get(subtitlesStyle.windowColor)));
			spinnerEdgeType.setSelection(edgeTypesAdapter.getPosition(edgeTypeToNameMap.get(subtitlesStyle.edgeType)));
			seekBarFontScale.setProgress((int) (subtitlesStyle.fontScale * 25));
		}

		findViewById(R.id.button_confirm_subtitles_style).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				if (dialogSubmitListener != null) {
					dialogSubmitListener.onDialogSubmit(SubtitleStyleDialog.this);
				}
			}
		});
	}

	public void setOnDialogSubmitListener(@Nullable OnDialogSubmitListener listener) {
		dialogSubmitListener = listener;
	}

	public int getBackgroundColor() {
		return getSpinnerColor(spinnerBackgroundColor);
	}

	public int getForegroundColor() {
		return getSpinnerColor(spinnerForegroundColor);
	}

	public int getEdgeColor() {
		return getSpinnerColor(spinnerEdgeColor);
	}

	public int getWindowColor() {
		return getSpinnerColor(spinnerWindowColor);
	}

	@SubtitlesStyle.EdgeType
	public int getEdgeType() {
		String edgeString = edgeTypesAdapter.getItem(spinnerEdgeType.getSelectedItemPosition());

		@SubtitlesStyle.EdgeType
		int edge = edgeNameToTypeMap.get(edgeString);

		return edge;
	}

	public float getFontScale() {
		return seekBarFontScale.getProgress() / 25f;
	}

	private int getSpinnerColor(Spinner s) {
		String colorName = colorsAdapter.getItem(s.getSelectedItemPosition());
		return colorNameToValueMap.get(colorName);
	}

	public String getFontFamily() {
		return fontFamilyAdapter.getItem(spinnerFontFamily.getSelectedItemPosition());
	}
}
