package com.castlabs.sdk.downloads;

import android.app.Activity;
import android.graphics.Rect;
import android.os.Bundle;
import com.google.android.material.snackbar.Snackbar;
import android.util.Log;
import android.view.ViewGroup;

import com.castlabs.android.player.PlayerView;
import com.castlabs.sdk.playerui.PlayerControllerProgressBar;
import com.castlabs.sdk.playerui.PlayerControllerView;
import com.castlabs.sdk.thumbs.DefaultThumbnailView;
import com.castlabs.sdk.thumbs.ThumbsPlugin;

public class PlaybackActivity extends Activity implements PlayerControllerView.SeekBarListener, DefaultThumbnailView.Callback {
	private static final String TAG = "PlaybackActivity";
	// This is the player view that we use to start playback
	private PlayerView playerView;

	private PlayerControllerView playerControllerView;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_simple_playback_demo);

		playerView = findViewById(R.id.player_view);
		playerControllerView = findViewById(R.id.player_controls);

		// This demo assumes that you send an intent to this Activity that contains the
		// playback information. We need to pass the intents bundle on to the PlayerController
		// to start playback. The open() method might throw an Exception in case the bundle
		// contains not all mandatory parameters or the parameters are malformed.
		if (getIntent() != null && getIntent().getExtras() != null) {
			try {
				playerView.getPlayerController().open(getIntent().getExtras());
			} catch (Exception e) {
				Log.e(TAG, "Error while opening player: " + e.getMessage(), e);
				Snackbar.make(playerView, "Error while opening player: " + e.getMessage(),
						Snackbar.LENGTH_INDEFINITE).show();
			}
		} else {
			Snackbar.make(playerView, "No intent specified", Snackbar.LENGTH_INDEFINITE).show();
		}
	}

	// Delegate the onStart event to the player views lifecycle delegate.
	// The delegate will make sure that the screen safer will be disabled and
	// the display will not go to sleep
	@Override
	protected void onStart() {
		super.onStart();
		playerView.getLifecycleDelegate().start(this);
	}

	// Delegate the onResume event to the player views lifecycle delegate.
	// The delegate ensures that the player recovers from a saved state. This needs to
	// be implemented to ensure the the user can for example go to the home screen and
	// come back to this activity.
	@Override
	protected void onResume() {
		super.onResume();
		// Bind the controller view and its listener
		playerControllerView.bind(playerView);
		playerControllerView.setSeekBarListener(this); // This is for Thumbnails

		PlayerControllerProgressBar progressBar = findViewById(R.id.progress_bar);
		progressBar.bind(playerView.getPlayerController());

		playerView.getLifecycleDelegate().resume();
	}

	// Delegate the onStop event to the player views lifecycle delegate.
	// We release the player when the activity is stopped. This will release all the player
	// resources and save the current playback state. Saving the state is required so the
	// onResume callback can recover properly.
	@Override
	protected void onStop() {
		super.onStop();
		// Unbind the player controller view and remove the listener
		playerControllerView.unbind();
		playerControllerView.setSeekBarListener(null);

		playerView.getLifecycleDelegate().releasePlayer(false);
	}
	// Required for Thumbnails
	@Override
	public void onSeekbarScrubbed(long positionUs, double seekBarProgressPercent) {
		// User interaction on the seekbar. Request Thumbnail rendering to the Thumbnail View
		ThumbsPlugin.ThumbnailViewComponent thumbsView = playerView.getComponent(
				ThumbsPlugin.ThumbnailViewComponent.class);
		if (thumbsView != null){
			thumbsView.show(positionUs, this);
		}
	}

	@Override
	public void getThumbnailRect(Rect output, DefaultThumbnailView.ThumbnailInfo info, boolean isSmallScreen) {
		// Fill the output Rect with the coordinates where we want the Thumbnail to be displayed
		ViewGroup videoView = playerView.getVideoView();
		output.set(
				videoView.getLeft(),
				videoView.getTop(),
				videoView.getRight(),
				videoView.getBottom());
	}

	@Override
	public void onSeekbarReleased() {
		// Seek bar has been released, we want to stop showing Thumbnails. For this we get the
		// Thumbnails View and call .hide() on it.
		final ThumbsPlugin.ThumbnailViewComponent thumbsView = playerView.getComponent(
				ThumbsPlugin.ThumbnailViewComponent.class);
		if (thumbsView != null){
			thumbsView.hide();
		}
	}
}
