package com.castlabs.sdk.hdcp_demo;

import androidx.annotation.NonNull;

import android.app.Activity;
import android.os.Bundle;
import android.os.Parcelable;
import android.util.Log;
import android.view.View;
import android.widget.Button;

import com.castlabs.android.player.AbstractPlayerListener;
import com.castlabs.android.player.PlayerConfig;
import com.castlabs.android.player.PlayerView;
import com.castlabs.android.player.SingleControllerPlaylist;
import com.castlabs.android.player.exceptions.CastlabsPlayerException;
import com.castlabs.sdk.playerui.PlayerControllerProgressBar;
import com.castlabs.sdk.playerui.PlayerControllerView;
import com.google.android.material.snackbar.Snackbar;

public class SingleControllerPlaylistDemo extends Activity {

	private static final String TAG = "PlaylistDemo";
	private static final String SAVED_STATE = "savedState";

	private PlayerView playerView;
	private PlayerControllerView playerControllerView;
	private PlayerControllerProgressBar progressBar;
	private SingleControllerPlaylist playlist;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_single_controller_playlist_demo);

		playerView = findViewById(R.id.playerView);
		playerControllerView = findViewById(R.id.playerControllerView);
		progressBar = findViewById(R.id.progressBar);

		Button btPrev = findViewById(R.id.btPrev);
		btPrev.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				playlist.previousItem();
			}
		});

		Button btNext = findViewById(R.id.btNext);
		btNext.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				playlist.nextItem();
			}
		});

		initPlaylist();
		startPlayback(savedInstanceState);
    }

	@Override
	protected void onStart() {
		super.onStart();
		playerView.getLifecycleDelegate().start(this);
	}

	@Override
	protected void onResume() {
		super.onResume();
		playerView.getLifecycleDelegate().resume();
		playerView.setPlayerController(playlist);
		playerControllerView.bind(playerView);
		progressBar.bind(playlist);
	}

	@Override
	protected void onStop() {
		super.onStop();
		playerView.getLifecycleDelegate().releasePlayer(false);
		playerControllerView.unbind();
		progressBar.unbind();
	}

	@Override
	public void onSaveInstanceState(@NonNull Bundle outState) {
		Bundle bundle = new Bundle();
		playerView.getPlayerController().saveState(bundle);
		outState.putBundle(SAVED_STATE, bundle);
		super.onSaveInstanceState(outState);
	}

	private void initPlaylist() {

		playlist = new SingleControllerPlaylist(this, new SingleControllerPlaylist.AbstractPlaylistListener() {
			@Override
			public void onItemChange(@NonNull PlayerConfig config) {
				Log.d(TAG, "onItemChange: " + config.contentUrl);
			}
		});

		playlist.addPlayerListener(new AbstractPlayerListener() {
			@Override
			public void onFatalErrorOccurred(@NonNull CastlabsPlayerException error) {
				showError("Error: " + error.getMessage());
			}

			@Override
			public void onError(@NonNull CastlabsPlayerException error) {
				showError("Error: " + error.getMessage());
			}
		});

		playerView.setPlayerController(playlist);
	}

	private void startPlayback(Bundle savedInstanceState) {
		try {
			if (savedInstanceState == null) {
				Parcelable[] configs = getIntent().getExtras().getParcelableArray(MainActivity.INTENT_PLAYLIST);
				playlist.open(configs);
			}
			else {
				Bundle bundle = savedInstanceState.getBundle(SAVED_STATE);
				playlist.open(bundle);
			}
		}
		catch (Exception e) {
			showError(e.getMessage());
		}
	}

	private void showError(String errorMessage) {
		Log.e(TAG, errorMessage);
		Snackbar.make(playerView, errorMessage, Snackbar.LENGTH_SHORT).show();
	}
}
