package com.castlabs.sdk.analytics_demo

import android.app.Activity
import android.app.Application
import android.os.Build
import android.os.Bundle
import com.castlabs.android.PlayerSDK
import com.castlabs.demoutils.ActivityInsetManager
import com.castlabs.sdk.adobe.AdobePlugin
import com.castlabs.sdk.analytics_demo.utils.DefaultLifecycleCallbacks
import com.castlabs.sdk.broadpeak.BroadpeakPlugin
import com.castlabs.sdk.conviva.ConvivaPlugin
import com.castlabs.sdk.ima.ImaPlugin
import com.castlabs.sdk.mux.MuxPlugin
import com.castlabs.sdk.nielsen.NielsenPlugin
import com.castlabs.sdk.vimond.VimondPlugin
import com.castlabs.sdk.youbora.YouboraPlugin
import com.castlabs.logutils.Log
import com.conviva.sdk.ConvivaAnalytics
import com.conviva.sdk.ConvivaSdkConstants
import com.npaw.NpawPlugin
import com.npaw.core.options.AnalyticsOptions
import org.json.JSONException
import org.json.JSONObject

class AnalyticsDemo : Application(), DefaultLifecycleCallbacks {

	companion object {
		private const val TAG = "AnalyticsDemo"
		private const val YOUBORA_CUSTOMER_KEY = "castlabsdev"
		private const val CONVIVA_CUSTOMER_KEY = "602ff71e0df67f735ca3bbec54f0bd8f59828ba5"
		private const val CONVIVA_TOUCHSTONE_GATEWAY = "https://602ff71e0df67f735ca3bbec54f0bd8f59828ba5.ts-testonly.conviva.com/"
		private const val BROADPEAK_ANALYTICS_URL = "http://bpk78.broadpeak-vcdn.com"
		private const val BROADPEAK_DOMAIN_NAMES = "pf6.broadpeak-vcdn.com"
		private const val MUX_CUSTOMER_KEY = "9gpohq8b4q3h14emcg5fc4lms"
		private const val NIELSEN_KEY = "P2C0BEB1A-A03C-4B09-8AA5-1BC5C7ECD138"
	}

	private var activityCounter = 0
	private var isChangingConfigurations = false
	private lateinit var youboraPlugin: NpawPlugin

	override fun onCreate() {
		super.onCreate()
		registerActivityLifecycleCallbacks(ActivityInsetManager());
		registerActivityLifecycleCallbacks(this)

		registerYouboraPlugin()
		registerVimondPlugin()
		registerConvivaPlugin()
		registerAdobePlugin()
		registerBroadpeakPlugin()
		registerMuxPlugin()
		registerNielsenPlugin()
		registerImaPlugin()

		PlayerSDK.init(applicationContext)
	}

	override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
		super.onActivityCreated(activity, savedInstanceState)
		youboraPlugin.appAnalytics.begin(activity::class.java.simpleName)
	}

	override fun onActivityStarted(activity: Activity) {
		super.onActivityStarted(activity)
		activityCounter++
		if (activityCounter == 1 && !isChangingConfigurations) {
			// App enters foreground
			ConvivaAnalytics.reportAppForegrounded()
		}
	}

	override fun onActivityStopped(activity: Activity) {
		super.onActivityStopped(activity)
		activityCounter--
		isChangingConfigurations = activity.isChangingConfigurations
		if (activityCounter == 0 && !isChangingConfigurations) {
			// App enters background
			ConvivaAnalytics.reportAppBackgrounded()
		}
	}

	private fun registerYouboraPlugin() {

		// By passing the Youbora "Plugin" instance to the SDK, Youbora
		// will link the video sessions with the application session.
		val analyticsOptions = AnalyticsOptions()
		this.youboraPlugin = NpawPlugin.Builder(this, YOUBORA_CUSTOMER_KEY)
			.setAnalyticsOptions(analyticsOptions)
			// Enable Youbora Logging
			.setLogLevel(com.npaw.core.util.extensions.Log.Level.VERBOSE)
			.build()

		// Register Youbora Plugin
		val youboraPlugin = YouboraPlugin(this.youboraPlugin)
		PlayerSDK.register(youboraPlugin)
	}

	private fun registerVimondPlugin() {
		val vimondPlugin = VimondPlugin()
		PlayerSDK.register(vimondPlugin)
	}

	private fun registerConvivaPlugin() {
		val convivaPlugin = ConvivaPlugin(CONVIVA_CUSTOMER_KEY)

		// Set settings
		val settings = HashMap<String, Any>()
		settings[ConvivaSdkConstants.GATEWAY_URL] = CONVIVA_TOUCHSTONE_GATEWAY
		settings[ConvivaSdkConstants.LOG_LEVEL] = ConvivaSdkConstants.LogLevel.DEBUG
		convivaPlugin.setSettings(settings)

		// Set device info
		val deviceInfo = HashMap<String, Any>()
		deviceInfo[ConvivaSdkConstants.DEVICEINFO.OPERATING_SYSTEM_VERSION] = Build.VERSION.SDK_INT.toString()
		deviceInfo[ConvivaSdkConstants.DEVICEINFO.DEVICE_BRAND] = Build.BRAND
		deviceInfo[ConvivaSdkConstants.DEVICEINFO.DEVICE_MANUFACTURER] = Build.MANUFACTURER
		deviceInfo[ConvivaSdkConstants.DEVICEINFO.DEVICE_MODEL] = Build.MODEL
		convivaPlugin.setDeviceInfo(deviceInfo)

		// Register Conviva Plugin
		PlayerSDK.register(convivaPlugin)
	}

	private fun registerAdobePlugin() {
		val adobePlugin = AdobePlugin(true)
		PlayerSDK.register(adobePlugin)
	}

	private fun registerBroadpeakPlugin() {
		val broadpeakPlugin = BroadpeakPlugin(
			BROADPEAK_ANALYTICS_URL,
			null,
			BROADPEAK_DOMAIN_NAMES,
		)
		PlayerSDK.register(broadpeakPlugin)
	}

	private fun registerMuxPlugin() {
		val muxPlugin = MuxPlugin(MUX_CUSTOMER_KEY)
		PlayerSDK.register(muxPlugin)
	}

	private fun registerNielsenPlugin() {
		val nielsenConfig = JSONObject()
		try {
			nielsenConfig.put(NielsenPlugin.CONFIG_KEY_APPID, NIELSEN_KEY)
			nielsenConfig.put(NielsenPlugin.CONFIG_KEY_APPNAME, R.string.app_name)
			nielsenConfig.put(NielsenPlugin.CONFIG_KEY_APPVERSION, BuildConfig.VERSION_NAME)
			nielsenConfig.put(NielsenPlugin.CONFIG_KEY_SFCODE, NielsenPlugin.SF_CODE_DCR)
		}
		catch (e: JSONException) {
			Log.e(TAG, "Error while setting up Nielsen config", e)
		}
		val nielsenPlugin = NielsenPlugin(nielsenConfig)
		PlayerSDK.register(nielsenPlugin)
	}

	private fun registerImaPlugin() {
		val imaPlugin = ImaPlugin()
		imaPlugin.setEnabled(true)
		PlayerSDK.register(imaPlugin)
	}
}
