package com.castlabs.sdk.demos;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;

import com.castlabs.android.player.AbstractPlayerListener;
import com.castlabs.android.player.PlayerController;
import com.castlabs.android.player.PlayerListener;
import com.castlabs.sdk.playerui.PlayerControllerProgressBar;
import com.castlabs.sdk.playerui.PlayerControllerView;
import com.castlabs.logutils.Log;
import com.google.android.material.snackbar.Snackbar;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.app.NotificationCompat;
import androidx.core.app.TaskStackBuilder;

import android.view.KeyEvent;
import android.view.View;

import com.castlabs.android.SdkConsts;
import com.castlabs.android.player.PlayerService;
import com.castlabs.android.player.PlayerView;

public class SimplePlayServiceDemo extends Activity {
	private static final String TAG = "SimplePlayServiceDemo";

	private static final String INTENT_PLAYER_CATEGORY = "com.castlabs.sdk.demos.intent.player";
	private static final String ACTION_PLAYER_PLAY = "com.castlabs.sdk.demos.action.play";
	private static final String ACTION_PLAYER_PAUSE = "com.castlabs.sdk.demos.action.pause";
	private static final String ACTION_PLAYER_RELEASE = "com.castlabs.sdk.demos.action.release";

	// ID that we use to identify the background notifications
	private static final int NOTIFICATION_ID = 1;
	private static final String NOTIFICATION_CHANNEL_ID = "1";
	private static final String NOTIFICATION_CHANNEL_NAME = "Playback notification";
	// We use this main handler to post delayed messaged to hide the system
	// controls after 3 seconds
	@NonNull
	private final Handler handler = new Handler();
	// This is the player view that we use to start playback
	@Nullable
	private PlayerView playerView;
	// This is the service binder that we get once we connected to the service
	// Initialize it during first call to onResume() and keep it until the activity is destroyed:
	// onStop() && isFinishing() OR onDestroy() or when the service is disconnected
	@Nullable
	private PlayerService.Binder playerServiceBinder;
	// We store the bundle that contains the playback information here
	// so the service can use it once connected
	@Nullable
	private Bundle playbackBundle;
	private boolean notificationCreated;
	// The service connection
	//
	// The connection is responsible to wither start playback or recover from background
	// playback
	@NonNull
	private final ServiceConnection playerServiceConnection = new ServiceConnection() {

		public void onServiceConnected(ComponentName name, IBinder service) {
			// Get the service binder and initialize the service
			playerServiceBinder = (PlayerService.Binder) service;
			// Initialize the service binder. This will return true if
			// playback was recovered from background.
			boolean backgrounded = playerServiceBinder.init(playerView);

			// From now on you can use playerView.getPlayerController() to interact with the controller
			if (!backgrounded) {
				if (playbackBundle != null) {
					// The player was not restored from background and we have a bundle to open.
					// This indicates that we came from an onCreate() call instead of resuming
					// an existing activity
					try {
						playerView.getPlayerController().open(playbackBundle);
					} catch (Exception e) {
						Snackbar.make(playerView,
								"Error while opening playback bundle: " + e.getMessage(),
								Snackbar.LENGTH_INDEFINITE).show();
					}
				} else {
					// The player was not restored from background and no bundle was passed.
					// In this case we delegate to the views lifecycle delegate to eventually
					// restore a session
					playerView.getLifecycleDelegate().resume();
				}
			}

			// Bind the controller view and its listener
			playerControllerView.bind(playerView);

			PlayerControllerProgressBar progressBar = (PlayerControllerProgressBar) findViewById(R.id.progress_bar);
			progressBar.bind(playerView.getPlayerController());

			playbackBundle = null;
		}

		public void onServiceDisconnected(ComponentName name) {
			playerServiceBinder.release(playerView, true);
			playerServiceBinder = null;
		}
	};
	private PlayerControllerView playerControllerView;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_simple_playback_demo);

		// Get the view components from the layout
		playerView = (PlayerView) findViewById(R.id.player_view);

		// Get the controller view
		playerControllerView = (PlayerControllerView) findViewById(R.id.player_controls);

		if (getIntent().hasExtra(NOTIFICATION_CHANNEL_NAME)) {
			notificationCreated = false;
		}

		// Hide the system toolbars
		//
		// We use the system UI flags to put this activity into a stable layout state
		// and hide the system toolbars. This ensures that we can use the full real estate
		// to show the video playback but avoid going into immersive fullscreen mode.
		//
		// With this setting, the system toolbars (the status bar and the navigation controls)
		// will become visible if the user interacts with the activity. In this demo, we
		// simply hide the controls again again after 3 seconds.
		hideSystemControls();
		getWindow().getDecorView().setOnSystemUiVisibilityChangeListener(
				new View.OnSystemUiVisibilityChangeListener() {
					@Override
					public void onSystemUiVisibilityChange(int visibility) {
						if ((visibility & View.SYSTEM_UI_FLAG_HIDE_NAVIGATION) == 0) {
							// The navigation bar is no longer hidden, so we hide it
							// again in a few seconds
							delayedHideSystemControls();
						}
					}
				});

		// In this demo, we are using a background service so we can not start interacting
		// with the player controller before the service is connected and initialised.
		// We connect the service in onResume() and store the playback bundle here
		// so the service can use it to start playback later.
		if (getIntent() != null) {
			playbackBundle = getIntent().getExtras();
		} else {
			Snackbar.make(playerView, "No intent specified", Snackbar.LENGTH_INDEFINITE).show();
		}
	}

	// Delegate the onStart event to the player views lifecycle delegate.
	// The delegate will make sure that the screen safer will be disabled and
	// the display will not go to sleep
	@Override
	protected void onStart() {
		super.onStart();
		playerView.getLifecycleDelegate().start(this);
	}

	// Because we are using a service, we are NOT delegating directly to the player views lifecycle
	// event. Instead, we make sure that a connection to the player service is established. The
	// binder implementation will then take care of initializing playback or to recover from
	// a background playback session (see the implementation of playerServiceConnection).
	@Override
	protected void onResume() {
		super.onResume();

		if (playerServiceBinder == null) {
			// The first call to onResume() where we bind to the service to get playerServiceBinder instance
			Intent serviceIntent = new Intent(this, PlayerService.class);
			bindService(serviceIntent, playerServiceConnection, Context.BIND_AUTO_CREATE);
		} else {
			// We have the playerServiceBinder instance already, proceed with bindings
			playerServiceBinder.init(playerView);
			// Bind the controller view and its listener
			playerControllerView.bind(playerView);

			PlayerControllerProgressBar progressBar = (PlayerControllerProgressBar) findViewById(R.id.progress_bar);
			progressBar.bind(playerView.getPlayerController());

			playbackBundle = null;
		}
	}

	// With the player service, we have to cover a few use cases when the activity is stopped.
	//
	// 1) If we have no service connection, we delegate to the lifecycle delegate of the
	//    view to release the player and any locks acquired when the activity started.
	//
	// 2) If isFinishing() returns true, we assume the user wants to actually leave the
	//    activity and not start background playback. In that case we release the player
	//    through the binder implementation
	// 3) If the activity is not finishing, we send the player to the background using the
	//    service binder implementation
	//
	// In both cases, we unbind from the service and reset the binder.
	@Override
	protected void onStop() {
		super.onStop();

		// Unbind the player controller view and remove the listener
		playerControllerView.unbind();
		playerControllerView.setSeekBarListener(null);

		if (playerServiceBinder == null) {
			// We have no service connection, so we release the player without background playback
			playerView.getLifecycleDelegate().releasePlayer(false);
		} else {
			if (!isFinishing()) {
				Bundle openContentBundle = new Bundle();
				playerView.getPlayerController().saveState(openContentBundle);
				// The activity is not finishing but left through other means, i.e. Home button.
				// In this case, we send the player to background and display a notification
				playerServiceBinder.releaseToBackground(
						playerView,
						NOTIFICATION_ID,
						createBackgroundNotification(playerView.getPlayerController().getPlayerState(), openContentBundle),
						true,
						new NotificationHandler(playerView.getPlayerController(), openContentBundle));
			} else {
				// We are not sending the player to background since the activity is finishing
				// so we fully release the player
				playerServiceBinder.release(playerView, true);
				unbindService(playerServiceConnection);
				playerServiceBinder = null;
			}
		}
	}

	@Override
	protected void onDestroy() {
		super.onDestroy();
		if (!notificationCreated) {
			// Activity is finishing, we choose to stop the background playback
			if (playerServiceBinder != null) {
				playerServiceBinder.killBackgroundPlayer(true);
				unbindService(playerServiceConnection);
				playerServiceBinder = null;
			}
		}
	}

	@Override
	public boolean dispatchKeyEvent(KeyEvent event) {
		boolean handled = false;
		if (playerControllerView != null && playerControllerView.getVisibility() == View.VISIBLE) {
			handled = playerControllerView.dispatchKeyEvent(event);
		} else { // Not visible
			if (event.getAction() == KeyEvent.ACTION_DOWN){
				handled = playerControllerView.dispatchKeyEvent(event);
			}
		}
		return handled || super.dispatchKeyEvent(event);
	}

	// Utility function that we use to create a dummy notification that is displayed when the
	// player is going to background
	private Notification createBackgroundNotification(@NonNull PlayerController.State state, @NonNull Bundle openContentBundle) {
		Intent targetIntent = new Intent(getApplicationContext(), getClass());
		targetIntent.putExtras(openContentBundle);

		TaskStackBuilder stackBuilder = TaskStackBuilder.create(getApplicationContext());
		stackBuilder.addParentStack(getClass());
		stackBuilder.addNextIntent(targetIntent);

		NotificationCompat.Builder builder = new NotificationCompat.Builder(getApplicationContext(), NOTIFICATION_CHANNEL_ID);
		builder.setAutoCancel(false);
		builder.setOngoing(true);
		builder.setContentTitle("Background Playback");
		builder.setContentText(openContentBundle.getString(SdkConsts.INTENT_URL));
		builder.setSmallIcon(R.drawable.notifications_icon);
		builder.setStyle(new androidx.media.app.NotificationCompat.MediaStyle());
		builder.setContentIntent(stackBuilder.getPendingIntent(0,PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_MUTABLE));

		if (state == PlayerController.State.Playing) {
			builder.addAction(
				new NotificationCompat.Action.Builder(
					R.drawable.cl_ic_pause,
					"Pause",
					PendingIntent.getService(
						getApplicationContext(),
						0,
						new Intent(getApplicationContext(), PlayerService.class)
							.setAction(ACTION_PLAYER_PAUSE)
							.addCategory(INTENT_PLAYER_CATEGORY),
						PendingIntent.FLAG_MUTABLE))
					.build());
		} else if (state == PlayerController.State.Pausing) {
			builder.addAction(
				new NotificationCompat.Action.Builder(
					R.drawable.cl_ic_play_arrow,
					"Play",
					PendingIntent.getService(
						getApplicationContext(),
						0,
						new Intent(getApplicationContext(), PlayerService.class)
							.setAction(ACTION_PLAYER_PLAY)
							.addCategory(INTENT_PLAYER_CATEGORY),
						PendingIntent.FLAG_MUTABLE))
					.build());
		}

		builder.addAction(
			new NotificationCompat.Action.Builder(
				R.drawable.cl_ic_close,
				"Stop",
				PendingIntent.getService(
					getApplicationContext(),
					0,
					new Intent(getApplicationContext(), PlayerService.class)
						.setAction(ACTION_PLAYER_RELEASE)
						.addCategory(INTENT_PLAYER_CATEGORY),
					PendingIntent.FLAG_MUTABLE))
				.build());

		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
			NotificationManager notificationManager = (NotificationManager)getSystemService(Context.NOTIFICATION_SERVICE);
			notificationManager.createNotificationChannel(new NotificationChannel(NOTIFICATION_CHANNEL_ID, NOTIFICATION_CHANNEL_NAME, NotificationManager.IMPORTANCE_DEFAULT));
		}

		notificationCreated = true;

		return builder.build();
	}

	// Utility method that hides the system controls and makes sure
	// the player view can use all available screen real estate.
	private void hideSystemControls() {
		final View decorView = getWindow().getDecorView();
		decorView.setSystemUiVisibility(
				View.SYSTEM_UI_FLAG_FULLSCREEN |
						View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN |
						View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION |
						View.SYSTEM_UI_FLAG_LAYOUT_STABLE |
						View.SYSTEM_UI_FLAG_HIDE_NAVIGATION);
	}

	private void delayedHideSystemControls() {
		handler.postDelayed(new Runnable() {
			@Override
			public void run() {
				hideSystemControls();
			}
		}, 3000);
	}

	private final class NotificationHandler implements PlayerService.NotificationHandler {
		@Nullable
		private PlayerController playerController;
		@NonNull
		private Bundle openContentBundle;
		@NonNull
		private PlayerListener playerListener = new AbstractPlayerListener() {
			@Override
			public void onStateChanged(@NonNull PlayerController.State state) {
				NotificationManager notificationManager = (NotificationManager)getSystemService(Context.NOTIFICATION_SERVICE);
				notificationManager.notify(NOTIFICATION_ID, createBackgroundNotification(state, openContentBundle));
			}
		};

		NotificationHandler(@NonNull PlayerController playerController, @NonNull Bundle openContentBundle) {
			this.playerController = playerController;
			this.openContentBundle = openContentBundle;
			this.playerController.addPlayerListener(playerListener);
		}

		@Override
		public void release() {
			if (this.playerController != null) {
				this.playerController.removePlayerListener(playerListener);
				this.playerController = null;
			}
		}

		@Override
		public boolean onIntent(@NonNull Intent intent) {
			boolean release = false;
			Log.d(TAG, "Player action: " + intent.getAction());
			if (intent.hasCategory(INTENT_PLAYER_CATEGORY)) {
				if (ACTION_PLAYER_RELEASE.equals(intent.getAction())) {
					// Make sure to come back to the launching activity (list of play assets) when foregrounding the app
					SimplePlayServiceDemo.this.finish();
					release = true;
				} else if (ACTION_PLAYER_PLAY.equals(intent.getAction())) {
					if (playerController != null) {
						playerController.play();
					}
				} else if (ACTION_PLAYER_PAUSE.equals(intent.getAction())) {
					if (playerController != null) {
						playerController.pause();
					}
				}
			}
			return release;
		}
	}
}
