package com.castlabs.sdk.demos.ads;

import android.content.Context;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.castlabs.android.adverts.Ad;
import com.castlabs.android.adverts.AdApi;
import com.castlabs.android.adverts.AdInterface;
import com.castlabs.android.player.PlayerController;
import com.castlabs.android.player.exceptions.CastlabsPlayerException;
import com.castlabs.sdk.demos.R;
import com.castlabs.utils.TimeUtils;
import com.google.android.exoplayer2.util.Log;

import java.util.Locale;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Example implementation of a custom advertisement UI. The AdUi hooks up to the events fired
 * by {@link com.castlabs.android.adverts.AdInterface.Listener} handles UI elements accordingly.
 */
public class AdUi extends RelativeLayout implements AdInterface.Listener {
	private Button skipButton;
	private ProgressBar progressBar;
	private TextView textProgress;

	private PlayerController playerController;
	@Nullable
	private AdApi api;
	private Ad currentAd;

	public AdUi(@NonNull Context context) {
		super(context);
		createViews();
	}

	public AdUi(@NonNull Context context, @Nullable AttributeSet attrs) {
		super(context, attrs);
		createViews();
	}

	public AdUi(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
		super(context, attrs, defStyleAttr);
		createViews();
	}

	private void createViews() {
		final LayoutInflater inflater = (LayoutInflater) getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		inflater.inflate(R.layout.custom_ad_ui, this);
		skipButton = findViewById(R.id.adSkipButton);
		textProgress = findViewById(R.id.adTimeProgress);
		progressBar = findViewById(R.id.adProgressBar);
		setLayoutParams(new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));

		skipButton.requestFocus();

		skipButton.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				if (api != null) {
					api.skipAd();
				}
			}
		});
	}

	public void bind(PlayerController playerController) {
		unbind();
		this.playerController = playerController;
		playerController.getAdInterface().addAdListener(this);
	}

	public void unbind() {
		if (playerController != null) {
			playerController.getAdInterface().removeAdListener(this);
			playerController = null;
		}
	}

	@Override
	public void onAdWillStart(@NonNull Ad ad) {

	}

	@Override
	public void onAdStarted(@NonNull Ad ad) {
		currentAd = ad;
		if (currentAd != null && currentAd.isUiDisabled) {
			// Trigger first update
			onAdPlaybackPositionChanged(0);
			setVisibility(VISIBLE);
			bringToFront();
		}
	}

	@Override
	public void onAdCompleted() {
		if (currentAd != null && currentAd.isUiDisabled) {
			setVisibility(GONE);
			currentAd = null;
		}
	}

	@Override
	public void onAdError(Ad ad, CastlabsPlayerException exception) {
		if (currentAd != null && currentAd.isUiDisabled) {
			setVisibility(GONE);
			currentAd = null;
		}
	}

	@Override
	public void onAdSkipped() {
		if (currentAd != null && currentAd.isUiDisabled) {
			setVisibility(GONE);
			currentAd = null;
		}
	}

	@Override
	public void onAdPlaybackPositionChanged(long playbackPositionMs) {
		if (currentAd == null) {
			return;
		}
		if (currentAd.isUiDisabled) {
			final long adTimeLeftSec = TimeUtils.ms2s(currentAd.durationMs - playbackPositionMs);

			final int percentage = (int) (playbackPositionMs * 100 / currentAd.durationMs);
			progressBar.setProgress(percentage);

			final String position = "Ad : (" + String.format(Locale.US, "%d:%02d", adTimeLeftSec / 60, adTimeLeftSec % 60) + ")";
			textProgress.setText(position);

			if (currentAd != null && currentAd.isSkippable && currentAd.skipTimeOffset > 0) {
				final double timeToSkip = currentAd.skipTimeOffset - TimeUtils.ms2s(playbackPositionMs);
				skipButton.setText(timeToSkip > 0 ?
						String.format(Locale.US, "%.0f", timeToSkip)
						: "Skip Ad");
				skipButton.setVisibility(VISIBLE);
				skipButton.requestFocus();
			} else {
				skipButton.setVisibility(GONE);
			}
		}
	}

	@Override
	public void onAdPaused() {

	}

	@Override
	public void onAdResumed(Ad ad) {

	}

	@Override
	public void onAdClicked(String clickthroughUrl) {

	}

	public void setApi(@Nullable AdApi adApi) {
		this.api = adApi;
	}
}
