package com.castlabs.sdk.analytics_demo.utils

import android.os.Bundle
import android.os.Parcelable
import com.adobe.mobile.MediaSettings
import com.castlabs.analytics.AnalyticsMetaData
import com.castlabs.android.SdkConsts
import com.castlabs.sdk.adobe.AdobePlugin
import com.castlabs.sdk.analytics_demo.BuildConfig
import com.castlabs.sdk.conviva.ConvivaAnalyticsSession
import com.castlabs.sdk.conviva.ConvivaPlugin
import com.castlabs.sdk.mux.MuxPlugin
import com.castlabs.sdk.youbora.YouboraPlugin
import com.mux.stats.sdk.core.CustomOptions
import com.mux.stats.sdk.core.model.*
import com.npaw.core.options.AnalyticsOptions

class BundleBuilder(isLive: Boolean, assetId: String) {

	companion object {
		const val APP_NAME = "Analytics Demo"
		const val PLAYER_NAME = "Castlabs Player"
		const val VIEWER_ID = "Castlabs Viewer Id"
		const val DEFAULT_RESOURCE = "Castlabs Resource"
		const val VIMOND_ASSET_ID = "VIMOND_ASSET_ID"
		const val VIMOND_SUB = "VIMOND_SUB"
	}

	private val bundle = Bundle()

	init {

		// General metadata
		var analyticsMetaData = AnalyticsMetaData(isLive, assetId)
		analyticsMetaData.viewerId = VIEWER_ID

		// Conviva metadata
		analyticsMetaData = addConvivaMetadata(analyticsMetaData)

		// Youbora metadata
		analyticsMetaData = addYouboraMetadata(analyticsMetaData, assetId)

		// Mux metadata
		analyticsMetaData = addMuxMetadata(analyticsMetaData, isLive, assetId)

		// Adobe metadata
		analyticsMetaData = addAdobeMetadata(analyticsMetaData, assetId)

		// Put the final metadata
		put(SdkConsts.INTENT_ANALYTICS_DATA, analyticsMetaData)
	}

	fun put(key: String, value: String): BundleBuilder {
		bundle.putString(key, value)
		return this
	}

	fun put(key: String, value: Long): BundleBuilder {
		bundle.putLong(key, value)
		return this
	}

	fun put(key: String, value: Int): BundleBuilder {
		bundle.putInt(key, value)
		return this
	}

	fun put(key: String, value: Boolean): BundleBuilder {
		bundle.putBoolean(key, value)
		return this
	}

	fun put(key: String, value: Parcelable): BundleBuilder {
		bundle.putParcelable(key, value)
		return this
	}

	fun get(): Bundle {
		return bundle
	}

	private fun addConvivaMetadata(analyticsMetaData: AnalyticsMetaData): AnalyticsMetaData {
		val convivaBundle = Bundle()
		convivaBundle.putString(ConvivaAnalyticsSession.META_KEY_APPLICATION_NAME, APP_NAME)
		convivaBundle.putString(ConvivaAnalyticsSession.META_KEY_DEFAULT_RESOURCE, DEFAULT_RESOURCE)
		val convivaCustomTags = Bundle()
		convivaCustomTags.putString("c3.app.version", BuildConfig.VERSION_NAME)
		convivaCustomTags.putString("sampleCustomTag", "sampleCustomValue")
		return ConvivaPlugin.createMetadata(convivaBundle, analyticsMetaData, convivaCustomTags)
	}

	private fun addYouboraMetadata(analyticsMetaData: AnalyticsMetaData, assetId: String): AnalyticsMetaData {
		val options = AnalyticsOptions()
		options.contentTitle = assetId
		return YouboraPlugin.createMetadata(analyticsMetaData, options)
	}

	private fun addMuxMetadata(analyticsMetaData: AnalyticsMetaData, isLive: Boolean,
							   assetId: String): AnalyticsMetaData {
		val videoData = CustomerVideoData()
		videoData.videoTitle = assetId
		videoData.videoLanguageCode = "en"
		val playerData = CustomerPlayerData()
		playerData.viewerUserId = "userId"
		val viewData = CustomerViewData()
		viewData.viewSessionId = "viewSessionId"
		val customData = CustomData()
		customData.customData1 = "customDimensionValue1"
		val customerData = CustomerData(playerData, videoData, viewData)
		customerData.customData = customData
		val customOptions = CustomOptions()
		return MuxPlugin.createMetadata(isLive, assetId, customerData, customOptions, analyticsMetaData)
	}

	private fun addAdobeMetadata(analyticsMetaData: AnalyticsMetaData, assetId: String): AnalyticsMetaData {
		val sampleLength = 100.0
		val playerId = "player-id"
		val mediaSettings = MediaSettings.settingsWith(assetId, sampleLength, PLAYER_NAME, playerId)
		return AdobePlugin.createMetadata(mediaSettings, analyticsMetaData)
	}
}
