package com.castlabs.sdk.conviva_demo;

import android.app.Activity;
import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;

import com.castlabs.analytics.AnalyticsSession;
import com.castlabs.android.player.PlayerView;
import com.castlabs.sdk.playerui.PlayerControllerProgressBar;
import com.castlabs.sdk.playerui.PlayerControllerView;
import com.google.android.material.snackbar.Snackbar;

public class ReleasablePlayerPlaybackDemo extends Activity {

	private AnalyticsSession analyticsSession = null;

	private static final String TAG = "ReleasablePlaybackDemo";
	private PlayerView playerView;

	private PlayerControllerView playerControllerView;
	private Button buttonPlayer;
	private boolean playerReleased;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_releasable_playback_demo);

		// Get the view components from the layout
		playerView = (PlayerView) findViewById(R.id.player_view);
		// Get the controller view
		playerControllerView = (PlayerControllerView) findViewById(R.id.player_controls);

		buttonPlayer = findViewById(R.id.buttonPlayer);
		buttonPlayer.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				if (playerReleased) {
					startPlayback();
				} else {
					stopPlayback();
				}
				playerReleased = !playerReleased;
				updatePlayerButton();
			}
		});

		startPlayback();

		playerReleased = false;
		updatePlayerButton();
	}

	private void updatePlayerButton() {
		buttonPlayer.setText(playerReleased? "Open player" : "Release player");
	}

	private void startPlayback() {
		// This demo assumes that you send an intent to this Activity that contains the
		// playback information. We need to pass the intents bundle on to the PlayerController
		// to start playback. The open() method might throw an Exception in case the bundle
		// contains not all mandatory parameters or the parameters are malformed.
		final Intent i = getIntent();
		if (i != null) {

			// Restore old analytics session
			if (analyticsSession != null) {
				playerView.getPlayerController().setAnalyticsSession(analyticsSession);
				analyticsSession = null;
			}
			try {
				playerView.getPlayerController().open(i.getExtras());
			} catch (Exception e) {
				Log.e(TAG, "Error while opening player: " + e.getMessage(), e);
				Snackbar.make(playerView, "Error while opening player: " + e.getMessage(),
						Snackbar.LENGTH_INDEFINITE).show();
			}
		} else {
			Snackbar.make(playerView, "No intent specified", Snackbar.LENGTH_INDEFINITE).show();
		}
	}

	private void stopPlayback() {
		// Keep analytics session
		analyticsSession = playerView.getPlayerController().getAnalyticsSession();
		if (analyticsSession != null) {
			analyticsSession.detachFromController();
		}
		// Release player
		playerView.getPlayerController().release();
	}

	// Delegate the onStart event to the player views lifecycle delegate.
	// The delegate will make sure that the screen safer will be disabled and
	// the display will not go to sleep
	@Override
	protected void onStart() {
		super.onStart();
		playerView.getLifecycleDelegate().start(this);
	}

	// Delegate the onResume event to the player views lifecycle delegate.
	// The delegate ensures that the player recovers from a saved state. This needs to
	// be implemented to ensure the the user can for example go to the home screen and
	// come back to this activity.
	@Override
	protected void onResume() {
		super.onResume();
		// Bind the controller view and its listener
		playerControllerView.bind(playerView);

		PlayerControllerProgressBar progressBar = (PlayerControllerProgressBar) findViewById(R.id.progress_bar);
		progressBar.bind(playerView.getPlayerController());

		playerView.getLifecycleDelegate().resume();
	}

	// Delegate the onStop event to the player views lifecycle delegate.
	// We release the player when the activity is stopped. This will release all the player
	// resources and save the current playback state. Saving the state is required so the
	// onResume callback can recover properly.
	@Override
	protected void onStop() {
		super.onStop();

		// Unbind the player controller view and remove the listener
		playerControllerView.unbind();

		playerView.getLifecycleDelegate().releasePlayer(false);
	}

	@Override
	protected void onDestroy() {
		super.onDestroy();
		// The activity is about to be destroyed and since the analytics session was detached
		// manually we need to stop it manually as well
		if (analyticsSession != null) {
			analyticsSession.stop();
			analyticsSession = null;
		}
	}

}
