//
//  CLRangedURL.h
//  mpl
//
//  Created by Emil Pettersson on 06/04/16.
//  Copyright © 2016 castLabs GmbH. All rights reserved.
//

#import <Foundation/Foundation.h>

@interface CLRangedURL : NSObject

/**
 * Start of the range
 */
@property (nonatomic) UInt64 offset;

/**
 * Length of the range
 * A value of 0 indicates no length is given, i.e. an infinite length (HTTP does not allow empty ranges anyway).
 */
@property (nonatomic) UInt64 length;

/**
 * String with the reference URL
 */
@property (nonnull, nonatomic) NSString* referenceURL;

/**
 * String with the base URL
 */
@property (nullable, nonatomic) NSString* baseURL;

/**
 * Create RangedURL instance
 * @param[in] offset Offset of first byte to read
 * @param[in] length Number of bytes to read
 * @param[in] referenceURL Reference URL
 * @param[in] baseURL Base URL (may be nil)
 */
+ (nullable instancetype)rangedURLWithOffset:(UInt64)offset
                                      length:(UInt64)length
                                referenceURL:(nonnull NSString*)referenceURL
                                     baseURL:(nullable NSString*)baseURL;

/**
 * Create RangedURL instance with plain URL
 * @param[in] referenceURL Reference URL
 * @param[in] baseURL Base URL (may be nil)
 */
+(nullable instancetype)rangedURLWithReferenceURL:(nonnull NSString*)referenceURL
                                          baseURL:(nullable NSString*)baseURL;

/**
 * Initialize RangedURL instance
 * @param[in] offset Offset of first byte to read
 * @param[in] length Number of bytes to read
 * @param[in] referenceURL Reference URL
 * @param[in] baseURL Base URL (may be nil)
 */
- (nullable instancetype)initWithOffset:(UInt64)offset
                                 length:(UInt64)length
                           referenceURL:(nonnull NSString*)referenceURL
                                baseURL:(nullable NSString*)baseURL;

/**
 * Initialise RangedURL instance with plain URL
 * @param[in] referenceURL Reference URL
 * @param[in] baseURL Base URL (may be nil)
 */
-(nullable instancetype)initWithReferenceURL:(nonnull NSString*)referenceURL
                                     baseURL:(nullable NSString*)baseURL;

/**
 * Is a range set that is not equivalent to 0-infinity
 */
- (BOOL)hasRange;

/**
 * Get the resolved final URL
 */
- (nullable NSURL*)resolveURL;

/**
 * Prepare a HEAD request with the resolved URL (does NOT include a range, even if there is one)
 */
- (nullable NSMutableURLRequest*)prepareHeadRequest;

/**
 * Prepare a request with the resolved URL & range (if there is one)
 */
- (nullable NSMutableURLRequest*)prepareRequest;

@end
